using System;
using System.Data;
using System.Collections;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using Method = System.Reflection.MethodBase;
using WRKEVENT_TABLE = gov.va.med.vbecs.Common.VbecsTables.WorkloadEvent;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for WorkloadEvent.
	/// </summary>
	public class WorkloadEvent : BaseBusinessObject, IRecordStatusCode
	{
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5028"> 
		///		<ExpectedInput>Valid workloadProcessGuid, integer representing the multiplier</ExpectedInput>
		///		<ExpectedOutput>Valid instance of a workloadevent object with 
		///		workload processguid and multiplier set to input valuest</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5033"> 
		///		<ExpectedInput>InValid</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///     Constructor for workload event class
		///     called by someone who knows the process to associate the event to upon creation
		/// </summary>		
		public WorkloadEvent(System.Guid workloadProcessGuid, int weightMultiplier)
		{
			
			this.MakeNew(workloadProcessGuid, weightMultiplier);
			this._dtWorkLoadEvent.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="5034"> 
		///		<ExpectedInput>Valid datarow with appropriate columns and values defined</ExpectedInput>
		///		<ExpectedOutput>An instance of a workloadevent class populated with datarow values</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="5035"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///     Constructor that takes a datarow to populate the workload event object
		/// </summary>
		public WorkloadEvent(System.Data.DataRow dtRow)
		{
			this._dtWorkLoadEvent = GetEmptyWorkloadEventDataTable();
			this._drWorkLoadEvent = this._dtWorkLoadEvent.NewRow();

			this.LoadFromDataRow(dtRow);
			this._dtWorkLoadEvent.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="5264"> 
		///		<ExpectedInput>Valid VBECS Process ID</ExpectedInput>
		///		<ExpectedOutput>Array list of workload event objects associated with VBECS process</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5265"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///     Gets all the workload associated with a given workload process, used by the generate workoad event function
		/// </summary>
		public static System.Collections.ArrayList GetWorkloadEventsByProcessID(int vbecsProcessID)
		{
			System.Collections.ArrayList al = new System.Collections.ArrayList();
			System.Data.DataTable dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID(vbecsProcessID);
			foreach(System.Data.DataRow dtRow in dt.Rows)
			{
				int multiplier = dtRow.IsNull(Common.VbecsTables.WorkloadProcess.WeightMultiplier) ? 1 : System.Convert.ToInt32( dtRow[Common.VbecsTables.WorkloadProcess.WeightMultiplier]);
				BOL.WorkloadEvent wrkEvent = new BOL.WorkloadEvent((System.Guid)dtRow[WRKEVENT_TABLE.WorkloadProcessGuid], 
					multiplier);
				al.Add(wrkEvent);
			}
			return al;
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/12/2007</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="8483"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8484"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		/// <summary>
		/// Generates workload data, allows you to have a misc trans type and still save patient pointer
		/// CR 2212
		/// </summary>
		/// <param name="workloadProcessId"></param>
		/// <param name="guids"></param>
		/// <param name="patientGuid"></param>
		/// <param name="transactionType"></param>
		/// <param name="negativeMultiplier"></param>
		/// <returns></returns>
		public static System.Data.DataTable GenerateWorkloadData(Common.WorkloadProcessID workloadProcessId, System.Guid [] guids, System.Guid patientGuid,
			Common.WorkloadTransactionType transactionType, bool negativeMultiplier)
		{
			//Create workload event objects from the process id
			//one to many events must be generated for each 
			//process that occurs (and event is one LMIP procedure tied to one process id)
			//this retrieves an array list of workload event objects that need to be created for each type of event
			System.Collections.ArrayList al = BOL.WorkloadEvent.GetWorkloadEventsByProcessID((int) workloadProcessId);
			System.Data.DataTable dtWorkload = null;
			int nCount = 0;
			nCount = guids.GetLength(0);
			for(int nCtr = 0; nCtr < nCount; nCtr++)
			{
				//**********************************************
				//Create workload event data
				//**********************************************
				foreach(object obj in al)
				{
					BOL.WorkloadEvent workloadEvent = (BOL.WorkloadEvent) obj;

					if(patientGuid != Guid.Empty)
					{
						workloadEvent.PatientGuid = patientGuid;
					}
					
					switch(transactionType)
					{
						case Common.WorkloadTransactionType.Unit:
							workloadEvent.BloodUnitGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Patient:
							workloadEvent.PatientGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Miscellanious:
							workloadEvent.MiscellanousGuid = guids[nCtr];
							break;
					}
					int multiplier = workloadEvent.Multiplier;
					if(negativeMultiplier)
					{
						workloadEvent.Multiplier = - multiplier;
					}
					else
					{
						workloadEvent.Multiplier = multiplier;
					}
#if NUNIT
						workloadEvent.VistaTechId = "53341";
#else
					workloadEvent.VistaTechId = Common.LogonUser.LogonUserDUZ;
#endif
					workloadEvent.WorkloadEventStatusCode = Common.WorkloadEventStatus.PendingProcessing;
					workloadEvent.RecordStatusCode = Common.RecordStatusCode.Active;
					workloadEvent.WorkloadEventGuid = System.Guid.NewGuid();
				} // foreach
				System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GetWorkloadEventDT(al);
				if(dtWorkload == null)
				{
					dtWorkload = dtWorkloadEvents.Clone();
				}
				foreach(System.Data.DataRow dtRow in dtWorkloadEvents.Rows)
				{
                    //*** Fortify Justified Code ***
                    //dtWorkload can't be null since it is checked earler (see above). 
                    System.Data.DataRow dtNewRow = dtWorkload.NewRow();
                    //*** Fortify Justified Code ***
                    dtNewRow.ItemArray = dtRow.ItemArray;
					dtWorkload.Rows.Add(dtNewRow);
				}
			}
			return  dtWorkload;
		}
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6691"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6692"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Generates workload for work
		/// </summary>
		/// <param name="workloadProcessId"></param>
		/// <param name="guids"></param>
		/// <param name="transactionType"></param>
		/// <param name="negativeMultiplier"></param>
		/// <returns></returns>
		public static System.Data.DataTable GenerateWorkloadData(Common.WorkloadProcessID workloadProcessId, System.Guid [] guids, 
			Common.WorkloadTransactionType transactionType, bool negativeMultiplier)
		{
			//Create workload event objects from the process id
			//one to many events must be generated for each 
			//process that occurs (and event is one LMIP procedure tied to one process id)
			//this retrieves an array list of workload event objects that need to be created for each type of event
			System.Collections.ArrayList al = BOL.WorkloadEvent.GetWorkloadEventsByProcessID((int) workloadProcessId);
			System.Data.DataTable dtWorkload = null;
			int nCount = 0;
			nCount = guids.GetLength(0);
			for(int nCtr = 0; nCtr < nCount; nCtr++)
			{
				//**********************************************
				//Create workload event data
				//**********************************************
				foreach(object obj in al)
				{
					BOL.WorkloadEvent workloadEvent = (BOL.WorkloadEvent) obj;
					
					switch(transactionType)
					{
						case Common.WorkloadTransactionType.Unit:
							workloadEvent.BloodUnitGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Patient:
							workloadEvent.PatientGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Miscellanious:
							workloadEvent.MiscellanousGuid = guids[nCtr];
							break;
					}
					int multiplier = workloadEvent.Multiplier;
					if(negativeMultiplier)
					{
						workloadEvent.Multiplier = - multiplier;
					}
					else
					{
						workloadEvent.Multiplier = multiplier;
					}
#if NUNIT
						workloadEvent.VistaTechId = "53341";
#else
					workloadEvent.VistaTechId = Common.LogonUser.LogonUserDUZ;
#endif
					workloadEvent.WorkloadEventStatusCode = Common.WorkloadEventStatus.PendingProcessing;
					workloadEvent.RecordStatusCode = Common.RecordStatusCode.Active;
					workloadEvent.WorkloadEventGuid = System.Guid.NewGuid();
				} // foreach
				System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GetWorkloadEventDT(al);
				if(dtWorkload == null)
				{
					dtWorkload = dtWorkloadEvents.Clone();
				}
				foreach(System.Data.DataRow dtRow in dtWorkloadEvents.Rows)
				{
                    //*** Fortify Justified Code ***
                    //dtWorkload can't be null since it is checked earler (see above). 
                    System.Data.DataRow dtNewRow = dtWorkload.NewRow();
                    //*** Fortify Justified Code ***
                    dtNewRow.ItemArray = dtRow.ItemArray;
					dtWorkload.Rows.Add(dtNewRow);
				}
			}
			return  dtWorkload;
		}

		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6689"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6690"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateWorkloadData
		/// </summary>
		/// <param name="workloadProcessId"></param>
		/// <param name="orderedTestGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="guids"></param>
		/// <param name="transactionType"></param>
		/// <param name="negativeMultiplier"></param>
		/// <returns></returns>
		public static System.Data.DataTable GenerateWorkloadData(Common.WorkloadProcessID workloadProcessId, System.Guid orderedTestGuid, 
			System.Guid orderedComponentGuid, System.Guid [] guids, Common.WorkloadTransactionType transactionType, bool negativeMultiplier)
		{
			//Create workload event objects from the process id
			//one to many events must be generated for each 
			//process that occurs (and event is one LMIP procedure tied to one process id)
			//this retrieves an array list of workload event objects that need to be created for each type of event
			System.Collections.ArrayList al = BOL.WorkloadEvent.GetWorkloadEventsByProcessID((int) workloadProcessId);
			System.Data.DataTable dtWorkload = null;
			int nCount = 0;
			nCount = guids.GetLength(0);
			for(int nCtr = 0; nCtr < nCount; nCtr++)
			{
				//**********************************************
				//Create workload event data
				//**********************************************
				foreach(object obj in al)
				{
					BOL.WorkloadEvent workloadEvent = (BOL.WorkloadEvent) obj;
					
					switch(transactionType)
					{
						case Common.WorkloadTransactionType.Unit:
							workloadEvent.BloodUnitGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Patient:
							workloadEvent.PatientGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Miscellanious:
							workloadEvent.MiscellanousGuid = guids[nCtr];
							break;
					}
					int multiplier = workloadEvent.Multiplier;
					if(negativeMultiplier)
					{
						workloadEvent.Multiplier = - multiplier;
					}
					else
					{
						workloadEvent.Multiplier = multiplier;
					}
					workloadEvent.VistaTechId = Common.LogonUser.LogonUserDUZ;
					workloadEvent.WorkloadEventStatusCode = Common.WorkloadEventStatus.PendingProcessing;
					workloadEvent.RecordStatusCode = Common.RecordStatusCode.Active;
					workloadEvent.WorkloadEventGuid = System.Guid.NewGuid();
					if(orderedTestGuid != Guid.Empty )
					{
						workloadEvent.OrderedTestGuid = orderedTestGuid;
					}
									
					if(orderedComponentGuid != Guid.Empty)
					{
						workloadEvent.OrderedComponentGuid = orderedComponentGuid;
					}
				}
				System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GetWorkloadEventDT(al);
				if(dtWorkload == null)
				{
					dtWorkload = dtWorkloadEvents.Clone();
				}
				foreach(System.Data.DataRow dtRow in dtWorkloadEvents.Rows)
				{
                    //*** Fortify Justified Code ***
                    //dtWorkload can't be null since it is checked earler (see above). 
                    System.Data.DataRow dtNewRow = dtWorkload.NewRow();
                    //*** Fortify Justified Code ***
                    dtNewRow.ItemArray = dtRow.ItemArray;
					dtWorkload.Rows.Add(dtNewRow);
				}
			}
			return  dtWorkload;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8197"> 
		///		<ExpectedInput>Parameters</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8198"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateWorkloadData
		/// </summary>
		/// <param name="workloadProcessId"></param>
		/// <param name="weightMultiplier"></param>
		/// <param name="orderedTestGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="guids"></param>
		/// <param name="transactionType"></param>
		/// <param name="negativeMultiplier"></param>
		/// <returns></returns>
		public static System.Data.DataTable GenerateWorkloadData(Common.WorkloadProcessID workloadProcessId, int weightMultiplier, System.Guid orderedTestGuid, 
			System.Guid orderedComponentGuid, System.Guid [] guids, Common.WorkloadTransactionType transactionType, bool negativeMultiplier)
		{
			//Create workload event objects from the process id
			//one to many events must be generated for each 
			//process that occurs (and event is one LMIP procedure tied to one process id)
			//this retrieves an array list of workload event objects that need to be created for each type of event
			System.Collections.ArrayList al = BOL.WorkloadEvent.GetWorkloadEventsByProcessID((int) workloadProcessId);
			System.Data.DataTable dtWorkload = null;
			int nCount = 0;
			nCount = guids.GetLength(0);
			for(int nCtr = 0; nCtr < nCount; nCtr++)
			{
				//**********************************************
				//Create workload event data
				//**********************************************
				foreach(object obj in al)
				{
					BOL.WorkloadEvent workloadEvent = (BOL.WorkloadEvent) obj;
					
					switch(transactionType)
					{
						case Common.WorkloadTransactionType.Unit:
							workloadEvent.BloodUnitGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Patient:
							workloadEvent.PatientGuid = guids[nCtr];
							break;
						case Common.WorkloadTransactionType.Miscellanious:
							workloadEvent.MiscellanousGuid = guids[nCtr];
							break;
					}
					int multiplier = workloadEvent.Multiplier;
					//CR 1987 KD - I multiplied weightMulitplier by multiplier coming from Vista to fix problem with
					//invalid workload total for ABID tests. It looks like this method is used only for ABID and this change
					//should not affect workload collection for other use cases.
					if(negativeMultiplier)
					{
						workloadEvent.Multiplier = - weightMultiplier * multiplier;
					}
					else
					{
						workloadEvent.Multiplier = weightMultiplier * multiplier;
					}
					workloadEvent.VistaTechId = Common.LogonUser.LogonUserDUZ;
					workloadEvent.WorkloadEventStatusCode = Common.WorkloadEventStatus.PendingProcessing;
					workloadEvent.RecordStatusCode = Common.RecordStatusCode.Active;
					workloadEvent.WorkloadEventGuid = System.Guid.NewGuid();
					if(orderedTestGuid != Guid.Empty )
					{
						workloadEvent.OrderedTestGuid = orderedTestGuid;
					}
									
					if(orderedComponentGuid != Guid.Empty)
					{
						workloadEvent.OrderedComponentGuid = orderedComponentGuid;
					}
				}
				System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GetWorkloadEventDT(al);
				if(dtWorkload == null)
				{
					dtWorkload = dtWorkloadEvents.Clone();
				}
				foreach(System.Data.DataRow dtRow in dtWorkloadEvents.Rows)
				{
                    //*** Fortify Justified Code ***
                    //dtWorkload can't be null since it is checked earler (see above). 
                    System.Data.DataRow dtNewRow = dtWorkload.NewRow();
                    //*** Fortify Justified Code ***
                    dtNewRow.ItemArray = dtRow.ItemArray;
					dtWorkload.Rows.Add(dtNewRow);
				}
			}
			return  dtWorkload;
		}

		/// <summary>
		///     
		/// </summary>
		/// <param name="vbecsProcessID" type="int">
		///     <para>
		///         
		///     </para>
		/// </param>
		/// <returns>
		///     A System.Data.DataTable value...
		/// </returns>
		private System.Data.DataTable GetWorkloadEvents(int vbecsProcessID)
		{
			System.Data.DataTable dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID(vbecsProcessID);
			return dt;
		}
		/// <summary>
		/// OnColumnChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}
		/// <summary>
		/// _dtWorkLoadEvent
		/// </summary>
		private System.Data.DataTable _dtWorkLoadEvent;
		/// <summary>
		/// _drWorkLoadEvent
		/// </summary>
		private System.Data.DataRow _drWorkLoadEvent;
		
		private System.Int32 _weightMultiplier;

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6695"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6696"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public method creates an empty data table for a workload event record
		/// </summary>
		public static System.Data.DataTable GetEmptyWorkloadEventDataTable()
		{
			System.Data.DataTable dtWorkLoadEvent = new System.Data.DataTable();
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.WorkloadEventGuid, typeof(System.Guid));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.WorkloadEventDate,typeof(System.DateTime));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.WorkloadProcessGuid,typeof(System.Guid));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.OrderedComponentGuid,typeof(System.Guid));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.OrderedTestGuid,typeof(System.Guid));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.BloodUnitGuid,typeof(System.Guid));	
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.PatientGuid,typeof(System.Guid));	
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.MiscellaneousGuid,typeof(System.Guid));	
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.WorkloadEventStatusCode,typeof(char));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.RecordSentIndicator,typeof(bool));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.ProcessedDate,typeof(System.DateTime));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.ErrorText,typeof(String));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.PceEncounterNumber,typeof(String));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.RecordStatusCode,typeof(char));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.DivisionCode,typeof(String));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.WeightMultiplier,typeof(System.Int32));
			dtWorkLoadEvent.Columns.Add(WRKEVENT_TABLE.VistaTechId,typeof(String));


						
			return dtWorkLoadEvent;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6693"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6694"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetWorkloadEventDT
		/// </summary>
		/// <param name="al"></param>
		/// <returns></returns>
		public static System.Data.DataTable GetWorkloadEventDT(System.Collections.ArrayList al)
		{
			System.Data.DataTable dt = BOL.WorkloadEvent.GetEmptyWorkloadEventDataTable();
			foreach(object obj in al)
			{
				BOL.WorkloadEvent workloadEvent = (BOL.WorkloadEvent) obj;
				System.Data.DataRow dtNewRow  = dt.NewRow();
				System.Data.DataRow dtRow = workloadEvent.LoadDataRowFromThis(dtNewRow);
				dtNewRow.ItemArray = dtRow.ItemArray;
				dt.Rows.Add(dtNewRow);
			}
			dt.TableName = Common.VbecsTables.WorkloadEvent.TableName;
			return dt;
		}

		/// <summary>
		///     Populates this instance of the object with the values from the datarow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadEventGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.WorkloadEventGuid))
				{
					this.WorkloadEventGuid = (System.Guid)dtRow[WRKEVENT_TABLE.WorkloadEventGuid];
				}
				else
				{
					this.WorkloadEventGuid = System.Guid.Empty;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadEventDate))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.WorkloadEventDate))
				{
					this.WorkloadEventDate = (System.DateTime)dtRow[WRKEVENT_TABLE.WorkloadEventDate];
				}
				else
				{
					this.WorkloadEventDate = DAL.VbecsDateTime.GetDivisionCurrentDateTime();
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadProcessGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.WorkloadProcessGuid))
				{
					this.WorkloadProcessGuid = (System.Guid)dtRow[WRKEVENT_TABLE.WorkloadProcessGuid];
				}
				else
				{
					this.WorkloadProcessGuid = System.Guid.Empty;
				}
			}
			
			
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.BloodUnitGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.BloodUnitGuid))
				{
					this.BloodUnitGuid = (System.Guid)dtRow[WRKEVENT_TABLE.BloodUnitGuid];
				}
				else
				{
					this.BloodUnitGuid = System.Guid.Empty;
				}
			}
			

			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.PatientGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.PatientGuid))
				{
					this.PatientGuid = (System.Guid)dtRow[WRKEVENT_TABLE.PatientGuid];
				}
				else
				{
					this.PatientGuid = System.Guid.Empty;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.MiscellaneousGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.MiscellaneousGuid))
				{
					this.MiscellanousGuid = (System.Guid)dtRow[WRKEVENT_TABLE.MiscellaneousGuid];
				}
				else
				{
					this.MiscellanousGuid = System.Guid.Empty;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.OrderedComponentGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.OrderedComponentGuid))
				{
					this.OrderedComponentGuid = (System.Guid)dtRow[WRKEVENT_TABLE.OrderedComponentGuid];
				}
				else
				{
					this.OrderedComponentGuid = System.Guid.Empty;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.OrderedTestGuid))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.OrderedTestGuid))
				{
					this.OrderedTestGuid = (System.Guid)dtRow[WRKEVENT_TABLE.OrderedTestGuid];
				}
				else
				{
					this.OrderedTestGuid = System.Guid.Empty;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadEventStatusCode))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.WorkloadEventStatusCode))
				{
					this.WorkloadEventStatusCode = Common.Utility.GetWorkloadEventStatusCodeEnumFromChar(dtRow[WRKEVENT_TABLE.WorkloadEventStatusCode].ToString().ToCharArray()[0]);
				}
				else
				{
					this.WorkloadEventStatusCode = Common.WorkloadEventStatus.Unknown;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.RecordSentIndicator))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.RecordSentIndicator))
				{
					this.RecordSentIndicator = (bool) dtRow[WRKEVENT_TABLE.RecordSentIndicator];
				}
				else
				{
					this.RecordSentIndicator = false;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.RecordStatusCode))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.RecordStatusCode))
				{
					this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[WRKEVENT_TABLE.RecordStatusCode].ToString());
				}
				else
				{
					this.RecordStatusCode = Common.RecordStatusCode.Unknown;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WeightMultiplier))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.WeightMultiplier))
				{
					this.Multiplier = System.Convert.ToInt32(dtRow[WRKEVENT_TABLE.WeightMultiplier].ToString());
				}
				else
				{
					this.Multiplier = 1;
				}
			}
			if(dtRow.Table.Columns.Contains(WRKEVENT_TABLE.VistaTechId))
			{
				if(!dtRow.IsNull(WRKEVENT_TABLE.VistaTechId))
				{
					this.VistaTechId = dtRow[WRKEVENT_TABLE.VistaTechId].ToString();
				}
				else
				{
					this.VistaTechId = string.Empty;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7977"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7978"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     LoadDataRowFromThis
		/// </summary>
		/// <returns>
		///     A System.Data.DataRow value...
		/// </returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadEventGuid))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventGuid] = this.WorkloadEventGuid;
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadEventDate))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventDate]  = this.WorkloadEventDate;
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadProcessGuid))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.WorkloadProcessGuid]= this.WorkloadProcessGuid;
			}
			
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.BloodUnitGuid))
			{
				if(this.BloodUnitGuid == System.Guid.Empty)
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.BloodUnitGuid] = System.DBNull.Value;
				}
				else
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.BloodUnitGuid] = this.BloodUnitGuid;
				}
			}
		
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.PatientGuid))
			{
				if(this.PatientGuid == System.Guid.Empty)
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.PatientGuid] = System.DBNull.Value;
				}
				else
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.PatientGuid] = this.PatientGuid;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.MiscellaneousGuid))
			{
				if(this.MiscellanousGuid == System.Guid.Empty)
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.MiscellaneousGuid] = System.DBNull.Value;
				}
				else
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.MiscellaneousGuid] = this.MiscellanousGuid;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.OrderedComponentGuid))
			{
				if(this.OrderedComponentGuid == System.Guid.Empty)
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.OrderedComponentGuid] = System.DBNull.Value;
				}
				else
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.OrderedComponentGuid] = this.OrderedComponentGuid;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.OrderedTestGuid))
			{
				if(this.OrderedTestGuid == System.Guid.Empty)
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.OrderedTestGuid] = System.DBNull.Value;
				}
				else
				{
					_drWorkLoadEvent[WRKEVENT_TABLE.OrderedTestGuid] = this.OrderedTestGuid;
				}
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WorkloadEventStatusCode))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventStatusCode] = Common.Utility.GetWorkloadEventStatusCodeCharFromEnum(this.WorkloadEventStatusCode);
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.RecordSentIndicator))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.RecordSentIndicator] = this.RecordSentIndicator;
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.RecordStatusCode))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.RecordSentIndicator))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.RecordSentIndicator] = this.RecordSentIndicator;
			}
			if (dtRow.Table.Columns.Contains(WRKEVENT_TABLE.DivisionCode))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}
			if(dtRow.Table.Columns.Contains(WRKEVENT_TABLE.WeightMultiplier))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.WeightMultiplier] = this.Multiplier;
			}
			if(dtRow.Table.Columns.Contains(WRKEVENT_TABLE.VistaTechId))
			{
				_drWorkLoadEvent[WRKEVENT_TABLE.VistaTechId] = this.VistaTechId;
			}
			
			return this._drWorkLoadEvent;
		}

		/// <summary>
		/// Logic that will populate the class 
		/// </summary>
		private void MakeNew(System.Guid workloadProcessGUID, int weightMultiplier)
		{
			this._dtWorkLoadEvent = GetEmptyWorkloadEventDataTable();
			this._drWorkLoadEvent = this._dtWorkLoadEvent.NewRow();
						
			this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventGuid] = System.Guid.Empty;
			this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventDate] = DAL.VbecsDateTime.GetDivisionCurrentDateTime();
			this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadProcessGuid] = workloadProcessGUID;
			this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventStatusCode] = Common.Utility.GetWorkloadEventStatusCodeCharFromEnum(Common.WorkloadEventStatus.Unknown);
			this._drWorkLoadEvent[WRKEVENT_TABLE.RecordSentIndicator] = false;
			this._drWorkLoadEvent[WRKEVENT_TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			this._drWorkLoadEvent[WRKEVENT_TABLE.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			
			this._dtWorkLoadEvent.Rows.Add(this._drWorkLoadEvent);
			this.Multiplier = weightMultiplier;
			
			this.IsDirty = true;
		}


		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6562"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6563"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// WorkloadEventGuid
		/// </summary>
		public System.Guid WorkloadEventGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.WorkloadEventGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6564"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6565"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// WorkloadProcessGuid
		/// </summary>
		public System.Guid WorkloadProcessGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.WorkloadProcessGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadProcessGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadProcessGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6566"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6567"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitGuid
		/// </summary>
		public System.Guid BloodUnitGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.BloodUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.BloodUnitGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.BloodUnitGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6568"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6569"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PatientGuid
		/// </summary>
		public System.Guid PatientGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.PatientGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.PatientGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.PatientGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6570"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6571"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Multiplier
		/// </summary>
		/// <remarks>
		///     Not using data row cause it is not a persistable property
		/// </remarks>
		public System.Int32 Multiplier
		{
			get
			{
				return this._weightMultiplier;
			}
			set
			{
				this._weightMultiplier = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6572"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6573"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderedComponentGuid
		/// </summary>
		public System.Guid OrderedComponentGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.OrderedComponentGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.OrderedComponentGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.OrderedComponentGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6574"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6575"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderedTestGuid
		/// </summary>
		public System.Guid OrderedTestGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.OrderedTestGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.OrderedTestGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.OrderedTestGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6576"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6577"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MiscellanousGuid
		/// </summary>
		public System.Guid MiscellanousGuid
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.MiscellaneousGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drWorkLoadEvent[WRKEVENT_TABLE.MiscellaneousGuid];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.MiscellaneousGuid] = value;
			}
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6578"> 
		///		<ExpectedInput>Valid RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6579"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RecordStatusCode
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.RecordStatusCode))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Unknown).ToString();
				}

				return Common.Utility.GetRecordStatusCodeFromString(_drWorkLoadEvent[WRKEVENT_TABLE.RecordStatusCode].ToString());
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(value);
			}
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6588"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6589"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PceEncounterNumber
		/// </summary>
		public string PceEncounterNumber
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.PceEncounterNumber))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.PceEncounterNumber] = String.Empty;
				}

				return this._drWorkLoadEvent[WRKEVENT_TABLE.PceEncounterNumber].ToString().Trim();
			}
			set
			{
				if (value.Length > 15)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						Method.GetCurrentMethod().Name, true), "15").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				this._drWorkLoadEvent[WRKEVENT_TABLE.PceEncounterNumber] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6590"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6591"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     String containing a description of an error if any that occurred
		///     during processing
		/// </summary>
		public string ErrorText
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.ErrorText))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.ErrorText] = String.Empty;
				}

				return this._drWorkLoadEvent[WRKEVENT_TABLE.ErrorText].ToString().Trim();
			}
			set
			{
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						Method.GetCurrentMethod().Name, true), "255").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				this._drWorkLoadEvent[WRKEVENT_TABLE.ErrorText] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8199"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8200"> 
		///		<ExpectedInput>Invalid sting</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Vista Tech ID(DUZ)
		/// </summary>
		public string VistaTechId
		{
			get
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.VistaTechId))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.VistaTechId] = String.Empty;
				}

				return this._drWorkLoadEvent[WRKEVENT_TABLE.VistaTechId].ToString().Trim();
			}
			set
			{
				if (value.Length > 19)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						Method.GetCurrentMethod().Name, true), "19").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				this._drWorkLoadEvent[WRKEVENT_TABLE.VistaTechId] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6580"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6581"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Property for workloadeventstatuscode.  Status can be Error,
		///     In Process, Pending Processing, Sucessfully Processed.
		/// </summary>
		public Common.WorkloadEventStatus WorkloadEventStatusCode
		{
			get 
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.WorkloadEventStatusCode))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventStatusCode] = Common.Utility.GetWorkloadEventStatusCodeCharFromEnum(Common.WorkloadEventStatus.Unknown);
				}

				return Common.Utility.GetWorkloadEventStatusCodeEnumFromChar(this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventStatusCode].ToString().ToCharArray()[0]);
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventStatusCode] = Common.Utility.GetWorkloadEventStatusCodeCharFromEnum(value);
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6582"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6583"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Property to set the workload event date.  This is the date the workload
		///     was generated.
		/// </summary>
		public System.DateTime WorkloadEventDate
		{
			get 
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.WorkloadEventDate))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventDate] = DAL.VbecsDateTime.GetDivisionCurrentDateTime();
				}
				return (System.DateTime) this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventDate];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.WorkloadEventDate] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6584"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6585"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Date the workload was processed
		/// </summary>
		public System.DateTime ProcessedDate
		{
			get 
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.ProcessedDate))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.ProcessedDate] = DAL.VbecsDateTime.GetDivisionCurrentDateTime();
				}
				return (System.DateTime) this._drWorkLoadEvent[WRKEVENT_TABLE.ProcessedDate];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.ProcessedDate] = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6586"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6587"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
    	/// <summary>
		///     Indicates whether or not the workload event has been sent to VistA
		/// </summary>
		public bool RecordSentIndicator
		{
			get 
			{
				if (this._drWorkLoadEvent.IsNull(WRKEVENT_TABLE.RecordSentIndicator))
				{
					this._drWorkLoadEvent[WRKEVENT_TABLE.RecordSentIndicator] = false;
				}
				return (bool) this._drWorkLoadEvent[WRKEVENT_TABLE.RecordSentIndicator];
			}
			set
			{
				this._drWorkLoadEvent[WRKEVENT_TABLE.RecordSentIndicator] = value;
			}
		}

        /// <summary>
        /// GetWorkloadProcessGuidByProcessID
        /// </summary>
        /// <param name="vbecsProcessID"></param>
        /// <returns></returns>
        public static System.Data.DataTable GetWorkloadProcessGuidByProcessID(int vbecsProcessID)
        {
            return DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID(vbecsProcessID);
        }

	}
}
